/**
  ******************************************************************************
  * @file    main.c
  * @author  MCU Application Team
  * @brief   Main program body
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2023 Puya Semiconductor Co.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by Puya under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2016 STMicroelectronics.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by ST under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  */

/* Includes ------------------------------------------------------------------*/
#include "main.h"
#include "py32t092xx_ll_Start_Kit.h"

/* Private define ------------------------------------------------------------*/
#define COUNTOF(__BUFFER__)   (sizeof(__BUFFER__) / sizeof(*(__BUFFER__)))
#define TXSTARTMESSAGESIZE    (COUNTOF(aTxStartMessage) - 1)
#define TXENDMESSAGESIZE      (COUNTOF(aTxEndMessage) - 1)

/* Private variables ---------------------------------------------------------*/
uint8_t aRxBuffer[12] = {0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff};
uint8_t aTxStartMessage[] = "\n\r LPUART Hyperterminal communication based on DMA\n\r Enter 12 characters using keyboard :\n\r";
uint8_t aTxEndMessage[] = "\n\r Example Finished\n\r";
__IO ITStatus LpuartReady = RESET;

/* Private user code ---------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
static void APP_SystemClockConfig(void);
static void APP_ConfigLpuart(void);
static void APP_ConfigDma(void);
static void APP_LpuartTransmit_DMA(LPUART_TypeDef *LPLPUARTx, uint8_t *pData, uint16_t Size);
static void APP_LpuartReceive_DMA(LPUART_TypeDef *LPLPUARTx, uint8_t *pData, uint16_t Size);
static void APP_WaitToReady(void);

/**
  * @brief  Main program.
  * @param  None
  * @retval int
  */
int main(void)
{
  /* Enable SYSCFG and PWR clock */
  LL_APB1_GRP2_EnableClock(LL_APB1_GRP2_PERIPH_SYSCFG);
  LL_APB1_GRP1_EnableClock(LL_APB1_GRP1_PERIPH_PWR);

  /* Configure system clock */
  APP_SystemClockConfig();

  /* Initialize LED */
  BSP_LED_Init(LED_GREEN);

  /* LPUART configuration */
  APP_ConfigLpuart();

 /* Start the transmission process */
  APP_LpuartTransmit_DMA(LPUART1, (uint8_t*)aTxStartMessage, TXSTARTMESSAGESIZE);
  APP_WaitToReady();
  
  /* Put LPUART peripheral in reception process */
  APP_LpuartReceive_DMA(LPUART1, (uint8_t *)aRxBuffer, 12);
  APP_WaitToReady();

  /* Send the received Buffer */
  APP_LpuartTransmit_DMA(LPUART1, (uint8_t*)aRxBuffer, 12);
  APP_WaitToReady();
  
  /* Send the End Message */
  APP_LpuartTransmit_DMA(LPUART1, (uint8_t*)aTxEndMessage, TXENDMESSAGESIZE);
  APP_WaitToReady();
  
  /* Turn on LED if test passes then enter infinite loop */
  BSP_LED_On(LED_GREEN);

  /* Infinite loop */
  while (1)
  {
  }
}

/**
  * @brief  Configure system clock
  * @param  None
  * @retval None
  */
static void APP_SystemClockConfig(void)
{
  /* Enable HSI */
  LL_RCC_HSI_Enable();
  while(LL_RCC_HSI_IsReady() != 1)
  {
  }

  /* Set AHB prescaler: HCLK = SYSCLK */
  LL_RCC_SetAHBPrescaler(LL_RCC_SYSCLK_DIV_1);

  /* Select HSISYS as system clock source */
  LL_RCC_SetSysClkSource(LL_RCC_SYS_CLKSOURCE_HSISYS);
  while(LL_RCC_GetSysClkSource() != LL_RCC_SYS_CLKSOURCE_STATUS_HSISYS)
  {
  }

  /* Set APB prescaler: PCLK = HCLK */
  LL_RCC_SetAPB1Prescaler(LL_RCC_APB1_DIV_1);
  LL_Init1msTick(8000000);

  /* Update the SystemCoreClock global variable(which can be updated also through SystemCoreClockUpdate function) */
  LL_SetSystemCoreClock(8000000);
}

/**
  * @brief  Wait transfer complete
  * @param  None
  * @retval None
  */
static void APP_WaitToReady(void)
{
  while (LpuartReady != SET);
  
  LpuartReady = RESET;
}

/**
  * @brief  LPUART1 configuration function
  * @param  None
  * @retval None
  */
static void APP_ConfigLpuart(void)
{
  LL_GPIO_InitTypeDef GPIO_InitStruct = {0};
  LL_LPUART_InitTypeDef LPUART_InitStruct = {0};

  /* Enable GPIOB clock */
  LL_IOP_GRP1_EnableClock(LL_IOP_GRP1_PERIPH_GPIOB);
  /* Enable LPUART1 clock */
  LL_APB1_GRP1_EnableClock(LL_APB1_GRP1_PERIPH_LPUART1);
  
  /* GPIOB configuration */
  /* Select PB8: Rx, PB9: Tx */
  GPIO_InitStruct.Pin        = (LL_GPIO_PIN_8 | LL_GPIO_PIN_9);
  /* Select alternate function mode */
  GPIO_InitStruct.Mode       = LL_GPIO_MODE_ALTERNATE;
  /* Set output speed */
  GPIO_InitStruct.Speed      = LL_GPIO_SPEED_FREQ_VERY_HIGH;
  /* Set output type to push pull */
  GPIO_InitStruct.OutputType = LL_GPIO_OUTPUT_PUSHPULL;
  /* Enable pull up */
  GPIO_InitStruct.Pull       = LL_GPIO_PULL_UP;
  /* Set alternate function to LPUART1 function  */
  GPIO_InitStruct.Alternate  = LL_GPIO_AF_12;
  /* Initialize GPIOB */
  LL_GPIO_Init(GPIOB,&GPIO_InitStruct);

  /* Set LPUART1 interrupt priority  */
  NVIC_SetPriority(UART1_LPUART1_IRQn, 0);
  /* Enable LPUART1 interrupt request */
  NVIC_EnableIRQ(UART1_LPUART1_IRQn);

  /* Configure DMA */
  APP_ConfigDma();

  /* Set LPUART feature */
  LPUART_InitStruct.PrescalerValue      = LL_LPUART_PRESCALER_DIV1;
  /* Set baud rate */
  LPUART_InitStruct.BaudRate            = 9600;
  /* set word length to 8 bits: Start bit, 8 data bits, n stop bits */
  LPUART_InitStruct.DataWidth           = LL_LPUART_DATAWIDTH_8B;
  /* 1 stop bit */
  LPUART_InitStruct.StopBits            = LL_LPUART_STOPBITS_1;
  /* Parity control disabled  */
  LPUART_InitStruct.Parity              = LL_LPUART_PARITY_NONE;
  /* Set transmit and receive direction */
  LPUART_InitStruct.TransferDirection   = LL_LPUART_DIRECTION_TX_RX;
  LPUART_InitStruct.HardwareFlowControl = LL_LPUART_HWCONTROL_NONE;

  /* Initialize LPUART */
  LL_LPUART_Init(LPUART1, &LPUART_InitStruct);
  LL_LPUART_Enable(LPUART1);
}

/**
  * @brief  DMA configuration function
  * @param  None
  * @retval None
  */
static void APP_ConfigDma(void)
{
  /* Enable DMA、syscfg clock */
  LL_AHB1_GRP1_EnableClock(LL_AHB1_GRP1_PERIPH_DMA1);
  LL_APB1_GRP2_EnableClock(LL_APB1_GRP2_PERIPH_SYSCFG);

  /* Configure DMA channel LL_DMA_CHANNEL_1 for transmission */
  LL_DMA_ConfigTransfer(DMA, LL_DMA_CHANNEL_1, LL_DMA_DIRECTION_MEMORY_TO_PERIPH | \
                      LL_DMA_MODE_NORMAL       | \
                      LL_DMA_PERIPH_FIX        | \
                      LL_DMA_MEMORY_INCREMENT  | \
                      LL_DMA_PDATAALIGN_BYTE   | \
                      LL_DMA_MDATAALIGN_BYTE   | \
                      LL_DMA_PRIORITY_0);
  
 
  /* Configure DMA channel LL_DMA_CHANNEL_2 for reception */
  LL_DMA_ConfigTransfer(DMA, LL_DMA_CHANNEL_2, LL_DMA_DIRECTION_PERIPH_TO_MEMORY | \
                      LL_DMA_MODE_NORMAL       | \
                      LL_DMA_PERIPH_FIX        | \
                      LL_DMA_MEMORY_INCREMENT  | \
                      LL_DMA_PDATAALIGN_BYTE   | \
                      LL_DMA_MDATAALIGN_BYTE   | \
                      LL_DMA_PRIORITY_1);

  /* Map the DMA request to LPUART1 */
  /* LPUART1_TX Corresponding channel LL_DMA_CHANNEL_1, LPUART1_RX Corresponding channel LL_DMA_CHANNEL_2 */
  LL_SYSCFG_SetDMARemap(DMA, LL_DMA_CHANNEL_1, LL_SYSCFG_DMA_MAP_LPUART1_TX);
  LL_SYSCFG_SetDMARemap(DMA, LL_DMA_CHANNEL_2, LL_SYSCFG_DMA_MAP_LPUART1_RX);
  
  /* Set interrupt priority */
  NVIC_SetPriority(DMA_Channel1_IRQn, 1);
   /* Enable interrupt */
  NVIC_EnableIRQ(DMA_Channel1_IRQn);
  
  /* Set interrupt priority */
  NVIC_SetPriority(DMA_Channel2_3_IRQn, 1);
  /* Enable interrupt */
  NVIC_EnableIRQ(DMA_Channel2_3_IRQn);
}

/**
  * @brief  LPUART transmit function
  * @param  LPUARTx：LPUART module
  * @param  pData：transmit buffer
  * @param  Size：Size of the transmit buffer
  * @retval None
  */
static void APP_LpuartTransmit_DMA(LPUART_TypeDef *LPUARTx, uint8_t *pData, uint16_t Size)
{  
  /* Configure DMA channel 1 */
  LL_DMA_DisableChannel(DMA, LL_DMA_CHANNEL_1);
  LL_DMA_ClearFlag_TC1(DMA);
  uint32_t *temp = (uint32_t *)&pData;
  LL_DMA_SetMemoryAddress(DMA, LL_DMA_CHANNEL_1, *(uint32_t *)temp);
  LL_DMA_SetPeriphAddress(DMA, LL_DMA_CHANNEL_1, (uint32_t)&(LPUARTx->TDR));
  LL_DMA_SetBlockLength(DMA, LL_DMA_CHANNEL_1, Size);
  LL_DMA_EnableIT_TC(DMA, LL_DMA_CHANNEL_1);
  LL_DMA_EnableChannel(DMA, LL_DMA_CHANNEL_1);

  /* Enable LPUART DMA channel for transmission */
  LL_LPUART_EnableDMAReq_TX(LPUARTx);
}

/**
  * @brief  LPUART receive function
  * @param  LPUARTx：LPUART module, can be LPUART1 or LPUART2
  * @param  pData：receive buffer
  * @param  Size：Size of the receive buffer
  * @retval None
  */
static void APP_LpuartReceive_DMA(LPUART_TypeDef *LPUARTx, uint8_t *pData, uint16_t Size)
{
  /*Configure DMA channel 2*/
  LL_DMA_DisableChannel(DMA, LL_DMA_CHANNEL_2);
  LL_DMA_ClearFlag_TC2(DMA);
  uint32_t *temp = (uint32_t *)&pData;
  LL_DMA_SetMemoryAddress(DMA, LL_DMA_CHANNEL_2, *(uint32_t *)temp);
  LL_DMA_SetPeriphAddress(DMA, LL_DMA_CHANNEL_2, (uint32_t)&LPUARTx->RDR);
  LL_DMA_SetBlockLength(DMA, LL_DMA_CHANNEL_2, Size);
  LL_DMA_EnableIT_TC(DMA, LL_DMA_CHANNEL_2);
  LL_DMA_EnableChannel(DMA, LL_DMA_CHANNEL_2);
  
  LL_LPUART_ClearFlag_ORE(LPUARTx);
  
  /* Enable LPUART DMA channel for transmission */
  LL_LPUART_EnableDMAReq_RX(LPUARTx);
}

/**
  * @brief  LPUART interrupt callback function
  * @param  LPUARTx：LPUART module, can be LPUART1 or LPUART2
  * @retval None
  */
void APP_LpuartIRQCallback(LPUART_TypeDef *LPUARTx)
{
  /* Transmission complete */
  if ((LL_LPUART_IsActiveFlag_TC(LPUARTx) != RESET) && (LL_LPUART_IsEnabledIT_TC(LPUARTx) != RESET))
  {
    LL_LPUART_DisableIT_TC(LPUARTx);
    LpuartReady = SET;
  
    return;
  }
}

/**
  * @brief  DMA channel 1 interrupt callback function for transmission
  * @param  None
  * @retval None
  */
void APP_DmaChannel1IRQCallback(void)
{
  if(LL_DMA_IsActiveFlag_TC1(DMA) == 1)
  {
    LL_DMA_ClearFlag_TC1(DMA);
    LL_LPUART_DisableDMAReq_TX(LPUART1);
    LL_LPUART_EnableIT_TC(LPUART1);
  }
}

/**
  * @brief  DMA interrupt callback function for reception
  * @param  None
  * @retval None
  */
void APP_DmaChannel2_3_IRQCallback(void)
{
  if(LL_DMA_IsActiveFlag_TC2(DMA) == 1)
  {
    LL_DMA_ClearFlag_TC2(DMA);
    LL_LPUART_DisableDMAReq_RX(LPUART1);
    LpuartReady = SET;
  }
}

/**
  * @brief  Error executing function.
  * @param  None
  * @retval None
  */
void APP_ErrorHandler(void)
{
  while (1)
  {
  }
}

#ifdef  USE_FULL_ASSERT
/**
  * @brief  Reports the name of the source file and the source line number
  *         where the assert_param error has occurred.
  * @param  file: pointer to the source file name
  * @param  line: assert_param error line source number
  * @retval None
  */
void assert_failed(uint8_t *file, uint32_t line)
{
  /* Users can add their own printing information as needed,
     for example: printf("Wrong parameters value: file %s on line %d\r\n", file, line) */
  /* Infinite loop */
  while (1)
  {
  }
}
#endif /* USE_FULL_ASSERT */

/************************ (C) COPYRIGHT Puya *****END OF FILE******************/
